<?php

namespace App\Http\Controllers\User;

use App\Actions\Gateway\SslCommerzPay\SslCommerz;
use App\Http\Controllers\Controller;
use App\Http\Requests\StoreBookingRequest;
use App\Models\Booking;
use App\Models\Package;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;

class BookingController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $bookings = user()->bookings()->with('package')->latest()->get();

        return view('user.booking.index', compact('bookings'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $data = session('booking');
        if (! $data || ! isset($data['package_id'])) {
            return redirect()->route('home');
        }
        $package = Package::findOrFail($data['package_id']);

        return view('user.booking.create', compact(['data', 'package']));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreBookingRequest $request)
    {
        // return
        // $request->travelers;
        $data = $request->validated();
        $session = session('booking');
        $package = Package::findOrFail($session['package_id']);
        $availableSeats = $package->max_people - $package->bookings->sum('peoples');

        if ($session['peoples'] > $availableSeats) {
            return back()->with('error', 'Sorry! Not enough seats available.');
        }
        DB::beginTransaction();
        try {
            $user = auth()->user();
            if (! $user) {
                $user = User::firstOrCreate([
                    'email' => $data['email'],
                ], array_merge($data, [
                    'password' => Hash::make('password'),
                ]));
                $user->assignRole('user');
                if (! auth()->check()) {
                    Auth::login($user);
                }

                $address = $user->addresses()->create(array_merge($data, [
                    'is_default' => true,
                ]));
            } else {
                $address = $user->addresses()->where('is_default', true)->first();
            }

            $booking = Booking::create([
                'package_id' => $session['package_id'],
                'user_id' => $user->id,
                'agent_id' => $package->agent_id,
                'address_id' => $address->id,
                'trnx_id' => uniqueId(12),
                'booking_date' => now(),
                'amount' => $this->calculateTotalAmount($package, $session),
                'status' => 'pending',
                'adult' => $session['adult'],
                'child' => $session['child'],
                'infant' => $session['infant'],
                'peoples' => $this->calculateTotalPeople($session),
                'payment_method' => $data['payment_method'],
            ]);

            // Add Travelers
            foreach ($request->travelers as $key => $travelers) {
                $booking->travelers()->createMany($travelers);
            }
            session()->forget('booking');
            DB::commit();

            if ($data['payment_method'] == 'online') {
                session()->put('booking_id', $booking->id);

                $sslz = new SslCommerz;
                $sslzResponse = $sslz->makePayment([
                    'total_amount' => $booking->amount,
                    'currency' => 'BDT',
                    'tran_id' => $booking->trnx_id,

                    // CUSTOMER INFORMATION
                    'cus_name' => $user->name,
                    'cus_email' => $user->email,
                    'cus_add1' => $address->address_one,
                    'cus_add2' => $address->address_two,
                    'cus_city' => $address->city->name,
                    'cus_state' => $address->state->name,
                    'cus_postcode' => $address->postal,
                    'cus_country' => $address->country->name,
                    'cus_phone' => $user->phone ?? '01945454545',
                    'cus_fax' => '',
                    'emi_option' => 0,

                    // SHIPMENT INFORMATION
                    'ship_name' => 'Store Test',
                    'ship_add1' => 'Dhaka',
                    'ship_add2' => 'Dhaka',
                    'ship_city' => 'Dhaka',
                    'ship_state' => 'Dhaka',
                    'ship_postcode' => '1000',
                    'ship_phone' => '',
                    'ship_country' => 'Bangladesh',

                    'shipping_method' => 'NO',
                    'product_name' => 'Computer',
                    'product_category' => 'Goods',
                    'product_profile' => 'physical-goods',
                ]);
            }

            return view('user.booking.success', compact('booking'));
        } catch (\Exception $e) {
            DB::rollBack();

            return $e->getMessage();

            return to_route('home');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Booking $booking)
    {
        if ($booking->user_id != user()->id) {
            abort(403, 'Sorry! You are not authorized to view this booking information.');
        }

        return view('user.booking.show', compact('booking'));
    }

    /**
     * Booking Session
     */
    public function session(Request $request, Package $package)
    {
        $validatedData = $request->validate([
            'adult' => 'required|numeric|min:0',
            'child' => 'required|numeric|min:0',
            'infant' => 'required|numeric|min:0',
        ]);

        $totalAmount = $this->calculateTotalAmount($package, $validatedData);
        $totalPeople = $this->calculateTotalPeople($validatedData);

        $bookingData = array_merge($validatedData, [
            'package_id' => $package->id,
            'amount' => $totalAmount,
            'peoples' => $totalPeople,
        ]);

        $request->session()->put('booking', $bookingData);

        return to_route('booking.create');
    }

    private function calculateTotalAmount(Package $package, array $data): float
    {
        return $package->adult_price * $data['adult'] +
            $package->child_price * $data['child'] +
            $package->infant_price * $data['infant'];
    }

    private function calculateTotalPeople(array $data): int
    {
        return $data['adult'] + $data['child'] + $data['infant'];
    }

    // Handle SSLCommerz response {success, cancel, fail, ipn}
    public function success(Request $request)
    {
        $booking = Booking::whereTrnxId($request->tran_id)->firstOrFail();
        if ($booking->payment_status != 'paid') {
            $booking->update([
                'status' => 'approved',
                'amount_paid' => $request->amount,
                'payment_status' => 'paid',
                'payment_method' => $request->card_type,
            ]);
        }

        if (! auth()->check()) {
            Auth::login($booking->user);
        }

        return view('user.booking.success', compact('booking'));
    }

    public function fail(Request $request)
    {
        return view('user.booking.fail');
    }

    public function ipn(Request $request)
    {
        $booking = Booking::findOrFail(session('booking_id'));
        $booking->update([
            'status' => 'ipn',
        ]);

        return view('user.booking.ipn', compact('booking'));
    }
}
